<?php

namespace Tobuli\Helpers\Payments\Gateways;

use Carbon\Carbon;
use Tobuli\Entities\Order;
use Illuminate\Http\Request;
use Tobuli\Entities\Subscription;

abstract class PaymentGateway
{
    public function storeConfig($request, $gateway)
    {
        settings('payments.gateways.' . $gateway, $request->active ? 1 : 0);
        settings('payments.' . $gateway,
            array_merge(settings('payments.' . $gateway), $request->except('_token', 'active'))
        );
    }

    public function renewSubscription($subscription, $expirationDate = null)
    {
        $subscription->renew($expirationDate);
    }

    public function isSubscriptionRenewed($subscription)
    {
        return $this->isSubscriptionActive($subscription);
    }

    public function getSubscriptionEnd($subscription)
    {
        return $subscription->expiration_date;
    }

    public function webhook(Request $request)
    {
        return response('', 404);
    }

    protected function gatewayName()
    {
        $name = (new \ReflectionClass($this))->getShortName();
        $name = str_replace('Gateway', '', $name);

        return snake_case($name);
    }

    protected function storeSubscription($user, Order $order, $gateway_id)
    {
        $subscription = Subscription::create([
            'user_id'    => $user->id,
            'gateway'    => $this->gatewayName(),
            'gateway_id' => $gateway_id,
            'order_id'   => $order->id,
        ]);

        return $subscription;
    }

    protected function activateSubscription($gateway_id, $options = [])
    {
        $subscription = Subscription::where('gateway_id', $gateway_id)->first();

        if ( ! $subscription) {
            throw new \Exception('Subscription not found for activation!');
        }

        $this->cancelSiblings($subscription);

        $subscription->activateEntity($options);
    }

    protected function cancelSiblings($subscription)
    {
        $siblingSubscriptions = Subscription::active()
            ->where('id', '!=', $subscription->id)
            ->where('user_id', $subscription->user_id)
            ->whereHas('order', function($query) use ($subscription) {
                $query->where('entity_id', $subscription->order->entity_id);
                $query->where('entity_type', $subscription->order->entity_type);
            })
            ->get();

        foreach ($siblingSubscriptions as $siblingSubscription) {
            try {
                $siblingSubscription->cancel();
            } catch (\Exception $e) {}
        }
    }

    /**
     * Compares two dates and checks if they make period with length of plan duration
     *
     * @param Carbon $expirationDateByPlan
     * @param Carbon $gatewayDate
     * @param int|null $planDurationInDays
     * @return boolean
     */
    protected function compareDatesByPlan(Carbon $expirationDateByPlan, Carbon $gatewayDate, $planDurationInDays)
    {
        if (empty($planDurationInDays)) {
            return $gatewayDate->gt($expirationDateByPlan);
        }

        $maxDiffPercentage = 30;

        $planDurationInMinutes = $planDurationInDays * 24 * 60;
        $diffInMinutes = $expirationDateByPlan->diffInMinutes($gatewayDate, false);
        $diffPercentage = ($diffInMinutes/$planDurationInMinutes) * 100;

        return $diffPercentage >= $maxDiffPercentage;
    }
}